// Arduino - Ethernet Relay Server
//
// Author: Jeróme Tüngler, Henri Gründer

#include <SPI.h>        // Serielle Kommunikation
#include <Ethernet.h>   // Bereitstellung des Servers
#include <SD.h>         // Zugriff auf SD-Karte
#include <TimerOne.h>   // Nutzung des integrierten Timers
#include <Wire.h>       // Kommunikation über I2C
#include <RTClib.h>     // Kommunikation mit RTC
#include <EEPROM.h>     // Benutzung des EEPROM

const byte r[] = {30, 33, 34, 37, 38, 41, 42, 45};        // Ports der Relais
const int tempSensor = A0;                                // Port des Temperatursensors
byte mac[] = {0x18, 0x97, 0x53, 0x2F, 0x43, 0xC4};  // Ethernet / TCP-IP Konfiguration
byte ip[] = {192, 168, 1, 123};
byte gateway[] = {192, 168, 1, 1};
byte subnet[] = {255, 255, 255, 0};

byte h = 0, m = 0, s = 0;  // Uhrzeit (Stunden, Minuten, )
boolean timer[8][48];      // Wann sollen die Relais An/Aus sein? (Auflösung 30 min)
boolean active[8];         // Sind die Timer aktiviert
EthernetClient client;     // ein Client, der die Seite anfordert
EthernetServer server(80); // Serverseite
String readString;         // übermittelte Anfrage
RTC_DS1307 RTC;            // RTC Baustein

void setup() {
  for (int i = 0; i < 8; i++) {
    pinMode(r[i], OUTPUT);
    digitalWrite(r[i], EEPROM.read(i));             // Gespeicherten Zustand wiederherstellen
    active[i] = EEPROM.read(392 + i);
  }
  for (int i = 0; i < 8; i++) {
    for (int j = 0; j < 48; j++) {
      timer[i][j] = EEPROM.read((i + j * 8) + 8);
    }
  }

  Ethernet.begin(mac, ip, gateway, subnet);  // Server starten
  server.begin();

  SD.begin(4);  // SD-Karten Zugriff an Port 4

  Timer1.initialize(1000000);    // Timer für die Uhrzeit starten
  Timer1.attachInterrupt(uhr);   // löst sekündlich aus
  Timer1.start();

  Wire.begin();  // RTC kommunikation via I2C
  RTC.begin();

  if (! RTC.isrunning()) {
    RTC.adjust(DateTime(2015, 5, 28, 0, 0, 0));  // Aktuelles Datum und Zeit setzen, falls die Uhr noch nicht läuft
  } else {
    uhrStellen();  // RTC Abfragen und interne Uhr danach stellen
  }
}

void loop() {
  client = server.available(); // Auf Anfrage warten
  if (client) {
    boolean currentLineIsBlank = true; // eine HTTP-Anfrage endet mit einer Leerzeile und einer neuen Zeile
    while (client.connected()) {
      if (client.available()) {
        char c = client.read(); // die Anfrage Zeichenweise lesen
        if (readString.length() < 100) {
          readString += c;
        }
        if (c == '\n' && currentLineIsBlank) {  // Client bereit für Datenempfang
          client.println("HTTP/1.1 200 OK");          // Default HHTP Header senden
          client.println("Content-Type: text/html");
          client.println("Connection: close");
          client.println();
          if (readString.indexOf("/?") > 0) {  // Es wurde ein Formular abgeschickt; Auswertung der Daten folgt
            openSD(SD.open("redirect.htm"));  // Weiterleitung zur Hauptseite senden
            client.stop();  // Vorzeitiges Beenden der Verbindung beschleunigt den Ladevorgang
            if (readString.indexOf("=Alle") > 0) {
              if (readString.indexOf("=AlleAn") > 0) {          // Alle Relais einschalten
                for (int i = 0; i < 8; i++) {
                  digitalWrite(r[i], LOW);
                  EEPROM.write(i, LOW);
                }
              } else if (readString.indexOf("=AlleAus") > 0) {  // Alle Relais ausschalten
                for (int i = 0; i < 8; i++) {
                  digitalWrite(r[i], HIGH);
                  EEPROM.write(i, HIGH);
                }
              }
            } else if (readString.indexOf("=An") > 0) {                 // bestimmtes Relais Einschalten
              char s = readString.charAt(readString.indexOf('?') + 1);  // welches Relais?
              digitalWrite(r[s - 49], LOW);
              EEPROM.write(s - 49, LOW);
            } else if (readString.indexOf("=Aus") > 0) {                // bestimmtes Relais Ausschalten
              char s = readString.charAt(readString.indexOf('?') + 1);  // welches Relais?
              digitalWrite(r[s - 49], HIGH);
              EEPROM.write(s - 49, HIGH);
            } else if (readString.indexOf("zeit") > 0) {  // Die Uhrzeit wurde eingestellt
              byte q = readString.indexOf('?');
              byte dz = readString.charAt(q + 3) - 48;
              byte de = readString.charAt(q + 4) - 48;
              byte mnz = readString.charAt(q + 8) - 48;
              byte mne = readString.charAt(q + 9) - 48;
              byte yz = readString.charAt(q + 13) - 48;
              byte ye = readString.charAt(q + 14) - 48;
              byte hz = readString.charAt(q + 18) - 48;
              byte he = readString.charAt(q + 19) - 48;
              byte mz = readString.charAt(q + 23) - 48;
              byte me = readString.charAt(q + 24) - 48;
              byte sz = readString.charAt(q + 28) - 48;
              byte se = readString.charAt(q + 29) - 48;
              byte dy = (dz * 10) + de;
              byte mn = (mnz * 10) + mne;
              byte yr = 2000 + (yz * 10) + ye;
              h = (hz * 10) + he;                         // Lokale Uhr stellen
              m = (mz * 10) + me;
              s = (sz * 10) + se;
              RTC.adjust(DateTime(yr, mn, dy, h, m, s));  // Zeit in die RTC eintragen
            } else if (readString.indexOf("time") > 0) {  // Ein Timer wurde eingestellt
              char zeitoochar[48];                        // An- und Auszeiten
              byte rel;                                   // Welches Relais
              byte q = readString.indexOf('?');
              q += 11;
              for (int i = 0; i < 48; i++) {              // An- / Aus-Werte ermitteln
                zeitoochar[i] = readString.charAt(q);
                q += 1;
              }
              q += 3;
              rel = readString.charAt(q) - 49;            // Relais ermitteln
              if (readString.indexOf("aktiv") > 0) {      // Wurde das Aktiv-Feld angehackt?
                active[rel] = 1;
                EEPROM.write(392 + rel, 1);
              } else {
                active[rel] = 0;
                EEPROM.write(392 + rel, 0);
              }
              for (int i = 0; i < 48; i++) {              // Ermittelte Werte speichern
                timer[rel][i] = zeitoochar[i] - 48;
                EEPROM.write((rel + i * 8) + 8, timer[rel][i]);
              }
            }
          } else if (readString.indexOf("/t") > 0) {    // Seite zum Stellen des Timers aufgerufen
            int rel = readString.charAt(readString.indexOf("/t") + 2) - 49;  //welches Relais?
            openSD(SD.open("timer.htm"));   // Seite anzeigen
            for (int i = 0; i < 48; i++) {  // Aktuelle Konfiguration übermitteln
              client.print(timer[rel][i]);
            }
            client.print("';setcolor();}");
            client.print("//--></script></body></html>");
          } else if (readString.indexOf("/uhrzeit") > 0) {  // Seite zum Stellen der Uhrzeit aufgerufen
            openSD(SD.open("uhrzeit.htm"));  // Seite anzeigen
          } else if (readString.indexOf("getRelaisState") > 0) {  // Status wurde abgefragt (Hintergrund)
            for (int i = 0; i < 8; i++) {        // Relaisstatus übermitteln
              if (digitalRead(r[i]) == HIGH) {
                client.print("0");
              } else {
                client.print("1");
              }
            }
            client.print("23.4");             // Temperatur übermitteln
            client.print("&a=");
            for (int i = 0; i < 8; i++) {        // Übermitteln, welche Timer aktiv sind
              client.print(active[i]);
            }
          } else {
            openSD(SD.open("index.htm"));  // bei keiner der obigen Aktionen die Übersichtsseite Anzeigen
            client.print(h);               // Eingestellte Uhrzeit übertragen
            client.print(",");
            client.print(m);
            client.print(",");
            client.print(s);
            client.println(");//--></script></body></html>");
          }
          break;
        }
        // Jede Zeile, die vom Client empfangen wird endet mit \r\n
        if (c == '\n') {        // Letztes Zeichen einer Zeile erhalten
          currentLineIsBlank = true;
        }
        else if (c != '\r') {   // Die empfangene Zeile ist nicht Leer
          currentLineIsBlank = false;
        }
      }
    }
    delay(1);
    client.stop();      // Aktuelle Übertragung beenden
    readString = "";
  }
}

void openSD(File f) {  // Abrufen einer Datei von der Mikro SD-Karte
  if (f) {
    while (f.available()) {
      client.write(f.read());
    }
    f.close();
  } else {
    client.write("Fehler beim Lesen von der SD-Karte");
  }
}

void uhr() {  // Wird sekündlich zum inkrementieren der Uhrzeit aufgerufen
  s ++;
  if (s == 60) {
    s = 0;
    m ++;
    if (m == 60) {
      m = 0;
      h ++;
      if (h == 24) {
        h = 0;
      }
    }
    if (m == 30) {  // Zur halben Stunde die Timer überprüfen
      for (int i = 0; i < 8; i++) {
        if (active[i] == 1) {
          if (timer[i][(h * 2) + 1] == 1) {
            digitalWrite(r[i], LOW);
            EEPROM.write(i, LOW);
          } else {
            digitalWrite(r[i], HIGH);
            EEPROM.write(i, HIGH);
          }
        }
      }
    } else if (m == 0) {  // Zur vollen Stunde die Timer überprüfen
      for (int i = 0; i < 8; i++) {
        if (active[i] == 1) {
          if (timer[i][h * 2] == 1) {
            digitalWrite(r[i], LOW);
            EEPROM.write(i, LOW);
          } else {
            digitalWrite(r[i], HIGH);
            EEPROM.write(i, HIGH);
          }
        }
      }
    }
  }
}

void uhrStellen() {  // Interne Uhr nach der RTC-Zeit stellen
  DateTime now = RTC.now();
  h = now.hour();
  m = now.minute();
  s = now.second();
}

float getTemp() {  // Temperatur vom Sensor auslesen
  int input = analogRead(tempSensor);
  float temp = ((input / 1024.0) * 500) - 50;
  return temp;
}
